<?php
// Start session
session_start();

// Initialize variables for messages
$error = '';
$success = '';

// If user is already logged in, redirect
if (isset($_SESSION['user_id'])) {
    header("Location: ../dashboard/");
    exit();
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Include database connection
    require_once '../conn/config.php';

    // Sanitize input
    $email = trim($_POST['email']);
    $password = $_POST['password'];

    // Validate input
    if (empty($email) || empty($password)) {
        $error = 'Email and password are required.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } else {
        // Check if user exists
        $stmt = $conn->prepare("SELECT id, password_hash, status, user_type FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 1) {
            $user = $result->fetch_assoc();

            // Verify password
            if (password_verify($password, $user['password_hash'])) {
                // Check account status
                if ($user['status'] === 'suspended') {
                    $error = 'Your account has been suspended. Please contact support.';
                } elseif ($user['status'] === 'rejected') {
                    $error = 'Your account was rejected. Please contact support.';
                } else {
                    // Login successful
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['user_email'] = $email;
                    $_SESSION['user_type'] = $user['user_type'];
                    $_SESSION['logged_in'] = true;

                    // Optional: Update last login
                    $update = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                    $update->bind_param("i", $user['id']);
                    $update->execute();
                    $update->close();

                    // Redirect to dashboard
                    header("Location: ../dashboard/");
                    exit();
                }
            } else {
                $error = 'Invalid password.';
            }
        } else {
            $error = 'No account found with that email.';
        }

        $stmt->close();
    }
    $conn->close();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <meta name="description" content="Log in to your MacraPay account to access secure payment services for individuals and businesses in Kenya. Accept cards and MPESA with ease.">
    <meta name="keywords" content="MacraPay, payment gateway, Kenya, MPESA, card payments, online payments, business payments, individual payments, login">
    <meta name="author" content="MacraPay Team">
    <meta name="robots" content="noindex, nofollow"> <!-- Prevent indexing until live -->

    <!-- Open Graph / Social Media Meta Tags -->
    <meta property="og:title" content="Login | MacraPay - Secure Payment Gateway">
    <meta property="og:description" content="Access your MacraPay account to manage payments, receive funds via MPESA and cards, and grow your business.">
    <meta property="og:type" content="website">
    <meta property="og:url" content="https://macrapay.com/login/">
    <meta property="og:image" content="https://macrapay.com/images/logo.png">
    <meta property="og:site_name" content="MacraPay">

    <!-- Twitter Card -->
    <meta name="twitter:card" content="summary_large_image">
    <meta name="twitter:title" content="Login | MacraPay - Secure Payment Gateway">
    <meta name="twitter:description" content="Securely log in to your MacraPay account and start accepting payments today.">
    <meta name="twitter:image" content="https://macrapay.com/images/logo.png">
    <meta name="twitter:site" content="@macrapay">

    <!-- Favicon -->
    <link rel="icon" type="image/png" href="../images/logo.png" />

    <!-- Font Awesome CDN -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" />

    <!-- Custom Styles -->
    <link rel="stylesheet" href="../css/styles.css" />

    <title>Login | MacraPay</title>
</head>
<body>

    <div class="nav">
        <div class="container">
            <a href="/" class="nav-logo">
                <img src="../images/logo.png" alt="MacraPay Logo"> MacraPay
            </a>
            <a href="../signup/" class="btn btn-secondary"><i class="fas fa-user-plus"></i> Sign Up</a>
        </div>
    </div>

    <div class="container">
        <div class="card">
            <img src="../images/logo.png" alt="MacraPay Logo" class="logo" />
            <h2 class="card-title"><i class="fas fa-sign-in-alt"></i> Log In to Your Account</h2>

            <!-- Display Messages -->
            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle"></i> <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?= htmlspecialchars($success) ?>
                </div>
            <?php endif; ?>

            <form id="loginForm" action="" method="POST">
                <div class="form-group input-icon">
                    <label for="email"><i class="fas fa-envelope"></i> Email Address</label>
                    <input 
                        type="email" 
                        id="email" 
                        name="email" 
                        placeholder="you@example.com" 
                        value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" 
                        required 
                    />
                </div>

                <div class="form-group input-icon">
                    <label for="password"><i class="fas fa-lock"></i> Password</label>
                    <input 
                        type="password" 
                        id="password" 
                        name="password" 
                        placeholder="••••••••" 
                        required 
                    />
                </div>

                <div class="form-group mt-1 text-right">
                    <a href="forgot-password.php" class="text-small" style="color: #007bff; font-size: 0.9rem;">
                        <i class="fas fa-question-circle"></i> Forgot password?
                    </a>
                </div>

                <div class="form-group mt-3">
                    <button type="submit" class="btn btn-primary btn-block">
                        <span class="btn-text"><i class="fas fa-sign-in-alt"></i> Log In</span>
                        <span class="loader" id="loader"></span>
                    </button>
                </div>
            </form>

            <p class="text-center mt-2">
                <small>
                    Don't have an account? <a href="../signup/">Sign up here</a>.
                </small>
            </p>
        </div>
    </div>

    <!-- Form UX: Show loader on submit -->
    <script>
        document.getElementById("loginForm").addEventListener("submit", function () {
            const btnText = document.querySelector(".btn-text");
            const loader = document.getElementById("loader");
            btnText.style.display = "none";
            loader.style.display = "inline-block";
        });
    </script>

</body>
</html>