<?php
// Start session and protect page
session_start();
if (!isset($_SESSION['user_id']) || !isset($_SESSION['logged_in']) || !$_SESSION['logged_in']) {
    header("Location: ../login/");
    exit();
}

// Include database connection
require_once '../conn/config.php';

// Initialize variables
$payments = [];
$error = '';

// Fetch payments for this user
$query = "
    SELECT 
        id,
        amount,
        currency,
        payment_method,
        reference,
        status,
        created_at 
    FROM payments 
    WHERE user_id = ? 
    ORDER BY created_at DESC 
    LIMIT 100
";

if ($stmt = $conn->prepare($query)) {
    $stmt->bind_param("i", $_SESSION['user_id']);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $payments[] = $row;
        }
    } else {
        $error = 'No payments found.';
    }
    $stmt->close();
} else {
    $error = 'Failed to load payments. Please try again.';
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>Payments | MacraPay</title>

    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" />

    <!-- Custom Styles -->
    <link rel="stylesheet" href="../css/styles.css" />
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="../images/logo.png" />
</head>
<body>
    <!-- Include Sidebar -->
    <?php include '../includes/sidebar.php'; ?>

    <!-- Main Content -->
    <div class="main-content">
        <?php include '../includes/header.php'; ?>

        <div class="content">
            <div class="container">
                <h1><i class="fas fa-credit-card"></i> Payments</h1>
                <p class="mb-2">View and search all payments received through MacraPay.</p>

                <!-- Search Bar -->
                <div class="form-group mb-2">
                    <label for="searchPayments"><i class="fas fa-search"></i> Search Payments</label>
                    <input 
                        type="text" 
                        id="searchPayments" 
                        class="form-control" 
                        placeholder="Search by reference, amount, date..." 
                    />
                </div>

                <?php if ($error): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle"></i> <?= htmlspecialchars($error) ?>
                    </div>
                <?php elseif (empty($payments)): ?>
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i> You haven't received any payments yet.
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table id="paymentsTable">
                            <thead>
                                <tr>
                                    <th>Date & Time</th>
                                    <th>Reference</th>
                                    <th>Amount</th>
                                    <th>Method</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($payments as $p): ?>
                                <tr data-reference="<?= htmlspecialchars($p['reference']) ?>"
                                    data-amount="<?= $p['amount'] ?>"
                                    data-date="<?= date('Y-m-d H:i', strtotime($p['created_at'])) ?>">
                                    <td>
                                        <?= date('M j, Y', strtotime($p['created_at'])) ?><br>
                                        <small><?= date('h:i A', strtotime($p['created_at'])) ?></small>
                                    </td>
                                    <td><strong><?= htmlspecialchars($p['reference'] ?? 'N/A') ?></strong></td>
                                    <td>
                                        <strong><?= htmlspecialchars($p['currency']) ?> <?= number_format($p['amount'], 2) ?></strong>
                                        <?php if ($p['currency'] !== 'KES'): ?>
                                            <div class="text-small" style="color: #007bff; font-size: 0.85rem;">
                                                <i class="fas fa-globe-africa"></i> USD
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <i class="fas 
                                            <?= $p['payment_method'] === 'card' ? 'fa-credit-card text-primary' : 'fa-mobile-alt text-success' ?>
                                        "></i>
                                        <?= ucfirst($p['payment_method']) ?>
                                        <?php if ($p['payment_method'] === 'mpesa'): ?>
                                            <div class="text-small" style="color: #28a745; font-size: 0.85rem;">
                                                <i class="fas fa-shield-alt"></i> KES
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="status 
                                            status-<?= 
                                                $p['status'] === 'success' ? 'success' : 
                                                ($p['status'] === 'pending' ? 'pending' : 'failed') 
                                            ?>
                                        ">
                                            <?= ucfirst($p['status']) ?>
                                        </span>
                                    </td>
                                    <td>
                                        <button 
                                            class="btn btn-sm btn-primary view-payment" 
                                            data-id="<?= (int)$p['id'] ?>"
                                            data-reference="<?= htmlspecialchars($p['reference']) ?>"
                                            data-amount="<?= $p['amount'] ?>"
                                            data-currency="<?= htmlspecialchars($p['currency']) ?>"
                                            data-method="<?= htmlspecialchars($p['payment_method']) ?>"
                                            data-status="<?= htmlspecialchars($p['status']) ?>"
                                            data-date="<?= date('M j, Y \a\t h:i A', strtotime($p['created_at'])) ?>"
                                            title="View Details">
                                            <i class="fas fa-eye"></i> View
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- 💡 VIEW MODAL -->
    <div id="viewModal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-credit-card"></i> Payment Details</h3>
                <span class="modal-close">&times;</span>
            </div>
            <div class="modal-body">
                <p><strong>Reference:</strong> <span id="modalReference"></span></p>
                <p><strong>Amount:</strong> <span id="modalAmount"></span></p>
                <p><strong>Method:</strong> <span id="modalMethod"></span></p>
                <p><strong>Status:</strong> <span id="modalStatus"></span></p>
                <p><strong>Date:</strong> <span id="modalDate"></span></p>
            </div>
            <div class="modal-footer">
                <button class="btn btn-secondary modal-close">Close</button>
            </div>
        </div>
    </div>

    <!-- 💡 MODAL STYLES (Add to styles.css later if needed) -->
    <style>
        .modal {
            position: fixed;
            z-index: 1001;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            background: #fff;
            border-radius: 12px;
            box-shadow: 0 8px 32px rgba(0,0,0,0.2);
            width: 90%;
            max-width: 500px;
            padding: 0;
            overflow: hidden;
        }

        .modal-header {
            padding: 1rem 1.5rem;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: #f8f9fa;
        }

        .modal-header h3 {
            margin: 0;
            color: #222;
            font-size: 1.3rem;
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #999;
            cursor: pointer;
            background: #f1f1f1;
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .modal-close:hover {
            color: #555;
            background: #e0e0e0;
        }

        .modal-body {
            padding: 1.5rem;
            line-height: 1.7;
            color: #444;
        }

        .modal-body p {
            margin-bottom: 1rem;
        }

        .modal-body strong {
            color: #222;
        }

        .modal-footer {
            padding: 1rem 1.5rem;
            border-top: 1px solid #eee;
            text-align: right;
        }
    </style>

    <!-- 💡 SEARCH & MODAL SCRIPT -->
    <script>
        // 🔍 Live Search Filter
        document.getElementById('searchPayments').addEventListener('keyup', function () {
            const term = this.value.toLowerCase();
            const rows = document.querySelectorAll('#paymentsTable tbody tr');

            rows.forEach(row => {
                const ref = row.dataset.reference.toLowerCase();
                const amount = row.dataset.amount;
                const date = row.dataset.date;

                const match = ref.includes(term) || 
                              amount.includes(term) || 
                              date.includes(term);

                row.style.display = match ? '' : 'none';
            });
        });

        // 👁️ View Modal
        const modal = document.getElementById('viewModal');
        const closeModal = () => modal.style.display = 'none';
        const modalCloseBtns = document.querySelectorAll('.modal-close, .modal-footer button');
        modalCloseBtns.forEach(btn => btn.addEventListener('click', closeModal));

        document.querySelectorAll('.view-payment').forEach(btn => {
            btn.addEventListener('click', function () {
                // Populate modal
                document.getElementById('modalReference').textContent = this.dataset.reference;
                document.getElementById('modalAmount').textContent = 
                    this.dataset.currency + ' ' + parseFloat(this.dataset.amount).toFixed(2);
                document.getElementById('modalMethod').textContent = 
                    this.dataset.method.charAt(0).toUpperCase() + this.dataset.method.slice(1);
                document.getElementById('modalStatus').textContent = 
                    this.dataset.status.charAt(0).toUpperCase() + this.dataset.status.slice(1);
                document.getElementById('modalDate').textContent = this.dataset.date;

                // Show modal
                modal.style.display = 'flex';
            });
        });

        // Close modal on click outside
        window.addEventListener('click', (e) => {
            if (e.target === modal) {
                closeModal();
            }
        });
    </script>
</body>
</html>