<?php
session_start();
if (!isset($_SESSION['user_id']) || !isset($_SESSION['logged_in']) || !$_SESSION['logged_in']) {
    header("Location: ../login/");
    exit();
}

// Get user balance
require_once '../conn/config.php';

$user_id = $_SESSION['user_id'];

// Get current balance
$balance_stmt = $conn->prepare("SELECT balance_after FROM transactions WHERE user_id = ? AND balance_after IS NOT NULL ORDER BY created_at DESC LIMIT 1");
$balance_stmt->bind_param("i", $user_id);
$balance_stmt->execute();
$balance_result = $balance_stmt->get_result();

$current_balance = 0;
if ($balance_result->num_rows > 0) {
    $row = $balance_result->fetch_assoc();
    $current_balance = $row['balance_after'];
}
$balance_stmt->close();
$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>Withdraw Funds | MacraPay</title>

    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" />

    <!-- Custom Styles -->
    <link rel="stylesheet" href="../css/styles.css" />

    <style>
        .payout-card {
            max-width: 600px;
            margin: 0 auto;
        }
        
        .balance-card {
            background: linear-gradient(135deg, #007bff, #0056b3);
            color: white;
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 2rem;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0,123,255,0.2);
        }
        
        .balance-amount {
            font-size: 2.5rem;
            font-weight: 700;
            margin: 0.5rem 0;
        }
        
        .balance-label {
            font-size: 1rem;
            opacity: 0.9;
        }
        
        .form-group label {
            font-weight: 600;
            color: #444;
        }
        
        .form-control {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #ddd;
            border-radius: 8px;
            font-size: 1rem;
            transition: border-color 0.3s;
        }
        
        .form-control:focus {
            border-color: #007bff;
            box-shadow: 0 0 0 3px rgba(0, 123, 255, 0.15);
        }
        
        .fee-info {
            background: #e7f3ff;
            border-left: 4px solid #007bff;
            padding: 1rem;
            border-radius: 0 8px 8px 0;
            margin: 1rem 0;
        }
        
        .destination-fields {
            display: none;
        }
        
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            justify-content: center;
            align-items: center;
        }
        
        .modal-content {
            background: #fff;
            border-radius: 12px;
            box-shadow: 0 8px 32px rgba(0,0,0,0.2);
            width: 90%;
            max-width: 500px;
            padding: 0;
            overflow: hidden;
            animation: modalSlideIn 0.3s ease;
        }
        
        @keyframes modalSlideIn {
            from { transform: translateY(-50px); opacity: 0; }
            to { transform: translateY(0); opacity: 1; }
        }
        
        .modal-header {
            padding: 1rem 1.5rem;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: #f8f9fa;
        }
        
        .modal-body {
            padding: 1.5rem;
            line-height: 1.7;
            color: #444;
            text-align: center;
        }
        
        .modal-body i {
            font-size: 4rem;
            margin-bottom: 1rem;
        }
        
        .modal-footer {
            padding: 1rem 1.5rem;
            border-top: 1px solid #eee;
            text-align: center;
        }
        
        .text-success {
            color: #28a745;
        }
        
        .text-danger {
            color: #dc3545;
        }
    </style>
</head>
<body>
    <!-- Include Sidebar -->
    <?php include '../includes/sidebar.php'; ?>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Include Header -->
        <?php include '../includes/header.php'; ?>

        <div class="content">
            <div class="container">
                <div class="payout-card">
                    <h1><i class="fas fa-money-bill-wave"></i> Withdraw Funds</h1>
                    <p class="text-center mb-3">Transfer money from your MacraPay balance to MPESA or Bank.</p>

                    <!-- Balance Display -->
                    <div class="balance-card">
                        <div class="balance-label">Available Balance</div>
                        <div class="balance-amount">KES <?= number_format($current_balance, 2) ?></div>
                        <div class="balance-label">Ready for withdrawal</div>
                    </div>

                    <!-- Status Messages -->
                    <div id="result"></div>

                    <!-- Payout Form -->
                    <form id="payoutForm">
                        <div class="form-group">
                            <label for="amount">Withdrawal Amount (KES)</label>
                            <input 
                                type="number" 
                                id="amount" 
                                class="form-control" 
                                step="0.01" 
                                min="10" 
                                max="<?= $current_balance ?>" 
                                placeholder="Enter amount" 
                                required 
                            />
                            <small class="text-muted">Minimum withdrawal: KES 10</small>
                        </div>

                        <div class="form-group">
                            <label for="destination_type">Destination Type</label>
                            <select id="destination_type" class="form-control" required>
                                <option value="">Select destination</option>
                                <option value="mpesa">MPESA Mobile Money</option>
                                <option value="bank">Bank Account</option>
                            </select>
                        </div>

                        <!-- MPESA Fields -->
                        <div class="destination-fields" id="mpesa_fields">
                            <div class="form-group">
                                <label for="phone">MPESA Phone Number</label>
                                <input 
                                    type="text" 
                                    id="phone" 
                                    class="form-control" 
                                    placeholder="254700123456" 
                                />
                                <small class="text-muted">Enter phone number starting with 254</small>
                            </div>
                        </div>

                        <!-- Bank Fields -->
                        <div class="destination-fields" id="bank_fields">
                            <div class="form-group">
                                <label for="bank_name">Bank Name</label>
                                <select id="bank_name" class="form-control">
                                    <option value="">Select Bank</option>
                                    <option value="KCB">Kenya Commercial Bank (KCB)</option>
                                    <option value="Equity">Equity Bank</option>
                                    <option value="Coop">Cooperative Bank</option>
                                    <option value="Absa">Absa Bank</option>
                                    <option value="Stanbic">Stanbic Bank</option>
                                    <option value="Standard_Chartered">Standard Chartered</option>
                                    <option value="NBK">National Bank of Kenya</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="bank_account_number">Bank Account Number</label>
                                <input 
                                    type="text" 
                                    id="bank_account_number" 
                                    class="form-control" 
                                    placeholder="Enter account number" 
                                />
                            </div>
                            
                            <div class="form-group">
                                <label for="bank_account_name">Account Holder Name</label>
                                <input 
                                    type="text" 
                                    id="bank_account_name" 
                                    class="form-control" 
                                    placeholder="Enter account holder name" 
                                />
                            </div>
                        </div>

                        <!-- Fee Information -->
                        <div class="fee-info">
                            <h4><i class="fas fa-info-circle"></i> Fees & Processing</h4>
                            <p><strong>MPESA Fee:</strong> 1% of withdrawal amount (minimum KES 10)</p>
                            <p><strong>Bank Transfer Fee:</strong> 1.5% of withdrawal amount (minimum KES 50)</p>
                            <p><strong>Processing Time:</strong> Instant for MPESA, 1-2 business days for Bank</p>
                            <p id="feeAmount">Fee will be calculated after you enter amount and select destination</p>
                        </div>

                        <div class="form-group mt-3">
                            <button type="submit" class="btn btn-primary btn-block" id="withdrawBtn">
                                <i class="fas fa-paper-plane"></i> Withdraw Funds
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Payout Status Modal -->
    <div id="payoutModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-money-bill-wave"></i> Withdrawal Status</h3>
                <button class="modal-close" type="button">&times;</button>
            </div>
            <div class="modal-body" id="modalBody">
                <!-- Content will be populated by JavaScript -->
            </div>
            <div class="modal-footer">
                <button id="closeModal" class="btn btn-primary">Back to Dashboard</button>
            </div>
        </div>
    </div>

    <script>
        // Toggle destination fields
        document.getElementById('destination_type').addEventListener('change', function() {
            const mpesaFields = document.getElementById('mpesa_fields');
            const bankFields = document.getElementById('bank_fields');
            
            document.querySelectorAll('.destination-fields').forEach(field => {
                field.style.display = 'none';
            });
            
            if (this.value === 'mpesa') {
                mpesaFields.style.display = 'block';
            } else if (this.value === 'bank') {
                bankFields.style.display = 'block';
            }
            
            // Recalculate fee
            calculateFee();
        });

        // Calculate fee in real-time
        function calculateFee() {
            const amount = parseFloat(document.getElementById('amount').value) || 0;
            const destinationType = document.getElementById('destination_type').value;
            
            if (amount > 0 && destinationType) {
                let fee = 0;
                let feeRate = 0;
                let minFee = 0;
                
                if (destinationType === 'mpesa') {
                    feeRate = 0.01; // 1%
                    minFee = 10;    // KES 10 minimum
                } else if (destinationType === 'bank') {
                    feeRate = 0.015; // 1.5%
                    minFee = 50;     // KES 50 minimum
                }
                
                fee = Math.max(amount * feeRate, minFee);
                const netAmount = amount - fee;
                
                document.getElementById('feeAmount').innerHTML = 
                    `<strong>You will receive:</strong> KES ${netAmount.toFixed(2)}<br>
                     <strong>Service fee:</strong> KES ${fee.toFixed(2)}`;
            } else {
                document.getElementById('feeAmount').textContent = 'Fee will be calculated after you enter amount and select destination';
            }
        }

        // Add event listeners for fee calculation
        document.getElementById('amount').addEventListener('input', calculateFee);
        document.getElementById('destination_type').addEventListener('change', calculateFee);

        // Format phone number
        document.getElementById('phone').addEventListener('input', function() {
            let phone = this.value.replace(/\D/g, '');
            if (phone.startsWith('0')) {
                phone = '254' + phone.substring(1);
            } else if (phone.startsWith('+254')) {
                phone = '254' + phone.substring(4);
            } else if (!phone.startsWith('254')) {
                phone = '254' + phone;
            }
            this.value = phone;
        });

        // Handle form submission
        document.getElementById('payoutForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const amount = parseFloat(document.getElementById('amount').value);
            const destinationType = document.getElementById('destination_type').value;
            const currentBalance = <?= $current_balance ?>;
            
            // Validation
            if (amount < 10) {
                showError('Minimum withdrawal amount is KES 10');
                return;
            }
            
            if (amount > currentBalance) {
                showError('Insufficient balance for this withdrawal');
                return;
            }
            
            if (!destinationType) {
                showError('Please select a destination type');
                return;
            }
            
            let payoutData = {
                amount: amount,
                destination_type: destinationType
            };
            
            // Validate destination-specific fields
            if (destinationType === 'mpesa') {
                let phone = document.getElementById('phone').value;
                if (!phone) {
                    showError('Please enter MPESA phone number');
                    return;
                }
                
                // Format phone
                phone = phone.replace(/\D/g, '');
                if (phone.startsWith('0')) {
                    phone = '254' + phone.substring(1);
                } else if (phone.startsWith('+254')) {
                    phone = '254' + phone.substring(4);
                } else if (!phone.startsWith('254')) {
                    phone = '254' + phone;
                }
                
                payoutData.phone = phone;
            } else if (destinationType === 'bank') {
                const bankName = document.getElementById('bank_name').value;
                const bankAccountNumber = document.getElementById('bank_account_number').value;
                const bankAccountName = document.getElementById('bank_account_name').value;
                
                if (!bankName || !bankAccountNumber || !bankAccountName) {
                    showError('Please fill in all bank account details');
                    return;
                }
                
                payoutData.bank_name = bankName;
                payoutData.bank_account_number = bankAccountNumber;
                payoutData.bank_account_name = bankAccountName;
            }
            
            // Calculate fee
            let feeRate = destinationType === 'mpesa' ? 0.01 : 0.015;
            let minFee = destinationType === 'mpesa' ? 10 : 50;
            const fee = Math.max(amount * feeRate, minFee);
            const netAmount = amount - fee;
            
            payoutData.fee = fee;
            payoutData.net_amount = netAmount;
            
            // Disable button and show loading
            const submitBtn = document.getElementById('withdrawBtn');
            const originalText = submitBtn.innerHTML;
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
            
            try {
                const response = await fetch('https://pay.macrapay.com/api/payouts/process.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    credentials: 'include',
                    body: JSON.stringify(payoutData)
                });
                
                const data = await response.json();
                
                if (data.status === 'success') {
                    let message = '';
                    if (destinationType === 'mpesa') {
                        message = `KES ${netAmount.toFixed(2)} will be sent to ${payoutData.phone}.<br>Transaction ID: ${data.payout_id}`;
                    } else {
                        message = `KES ${netAmount.toFixed(2)} will be sent to ${payoutData.bank_account_name}'s account.<br>Transaction ID: ${data.payout_id}`;
                    }
                    showPayoutModal('success', 'Withdrawal Initiated!', message);
                } else {
                    showPayoutModal('failed', 'Withdrawal Failed', data.error || 'Unable to process withdrawal');
                }
            } catch (error) {
                showPayoutModal('failed', 'Connection Error', 'Unable to connect to server. Please try again.');
            } finally {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            }
        });
        
        function showError(message) {
            document.getElementById('result').innerHTML = `
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle"></i> ${message}
                </div>
            `;
        }
        
        function showPayoutModal(status, title, message) {
            const modal = document.getElementById('payoutModal');
            const modalBody = document.getElementById('modalBody');
            
            let icon = 'fa-info-circle';
            let iconClass = '';
            
            if (status === 'success') {
                icon = 'fa-check-circle';
                iconClass = 'text-success';
            } else {
                icon = 'fa-times-circle';
                iconClass = 'text-danger';
            }
            
            modalBody.innerHTML = `
                <i class="fas ${icon} ${iconClass}"></i>
                <h3>${title}</h3>
                <p>${message}</p>
            `;
            
            modal.style.display = 'flex';
        }
        
        // Modal close handlers
        document.querySelector('.modal-close').addEventListener('click', function() {
            document.getElementById('payoutModal').style.display = 'none';
        });
        
        document.getElementById('closeModal').addEventListener('click', function() {
            document.getElementById('payoutModal').style.display = 'none';
            window.location.href = '../dashboard/';
        });
        
        // Close modal on click outside
        window.addEventListener('click', function(e) {
            const modal = document.getElementById('payoutModal');
            if (e.target === modal) {
                modal.style.display = 'none';
            }
        });
    </script>
</body>
</html>