<?php
// Start session (optional: if you want to track who created the request)
session_start();
if (!isset($_SESSION['user_id']) || !isset($_SESSION['logged_in']) || !$_SESSION['logged_in']) {
    header("Location: ../login/");
    exit();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>Request Payment | MacraPay</title>

    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" />

    <!-- Custom Styles -->
    <link rel="stylesheet" href="../css/styles.css" />
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="../images/logo.png" />

    <style>
        /* Form Enhancements */
        .form-group label {
            font-weight: 600;
            color: #444;
        }

        .form-control {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #ddd;
            border-radius: 8px;
            font-size: 1rem;
            transition: border-color 0.3s;
        }

        .form-control:focus {
            border-color: #007bff;
            box-shadow: 0 0 0 3px rgba(0, 123, 255, 0.15);
        }

        .btn {
            padding: 0.75rem 1.5rem;
            font-size: 1rem;
        }

        /* Card styling */
        .receive-card {
            max-width: 600px;
            margin: 0 auto;
        }

        .receive-icon {
            font-size: 3rem;
            color: #007bff;
            margin-bottom: 1rem;
        }

        .alert {
            margin-bottom: 1rem;
        }

        /* Modal Styles */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            background: #fff;
            border-radius: 12px;
            box-shadow: 0 8px 32px rgba(0,0,0,0.2);
            width: 90%;
            max-width: 500px;
            padding: 0;
            overflow: hidden;
            animation: modalSlideIn 0.3s ease;
        }

        @keyframes modalSlideIn {
            from { transform: translateY(-50px); opacity: 0; }
            to { transform: translateY(0); opacity: 1; }
        }

        .modal-header {
            padding: 1rem 1.5rem;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
            background: #f8f9fa;
        }

        .modal-header h3 {
            margin: 0;
            color: #222;
            font-size: 1.3rem;
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 1.5rem;
            color: #999;
            cursor: pointer;
            width: 36px;
            height: 36px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .modal-close:hover {
            color: #555;
            background: #e0e0e0;
        }

        .modal-body {
            padding: 1.5rem;
            line-height: 1.7;
            color: #444;
            text-align: center;
        }

        .modal-body i {
            font-size: 4rem;
            margin-bottom: 1rem;
        }

        .modal-body .success {
            color: #28a745;
        }

        .modal-body .failed {
            color: #dc3545;
        }

        .modal-footer {
            padding: 1rem 1.5rem;
            border-top: 1px solid #eee;
            text-align: center;
        }

        .progress-container {
            margin: 1rem 0;
        }

        .progress {
            height: 10px;
            background: #e9ecef;
            border-radius: 5px;
            overflow: hidden;
        }

        .progress-bar {
            height: 100%;
            background: #007bff;
            width: 0%;
            transition: width 0.3s ease;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .form-row {
                flex-direction: column;
                gap: 0;
            }
            .form-col {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <!-- Include Sidebar -->
    <?php include '../includes/sidebar.php'; ?>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Include Header -->
        <?php include '../includes/header.php'; ?>

        <div class="content">
            <div class="container">
                <div class="receive-card">
                    <h1><i class="fas fa-hand-holding-usd"></i> Request Payment</h1>
                    <p class="text-center mb-3">Fill in the details to generate a payment request.</p>

                    <!-- Status Messages -->
                    <div id="result"></div>

                    <form id="paymentForm">
                        <div class="form-group">
                            <label for="amount">Amount</label>
                            <input 
                                type="number" 
                                id="amount" 
                                class="form-control" 
                                step="0.01" 
                                placeholder="100.00" 
                                required 
                            />
                        </div>

                        <div class="form-group">
                            <label for="currency">Currency</label>
                            <select id="currency" class="form-control" required>
                                <option value="KES">KES (Kenyan Shilling)</option>
                                <option value="USD">USD (US Dollar)</option>
                            </select>
                            <small id="currencyNote" style="color: #007bff; display: none;">
                                <i class="fas fa-info-circle"></i> USD only available for card payments.
                            </small>
                        </div>

                        <div class="form-group">
                            <label for="method">Payment Method</label>
                            <select id="method" class="form-control" required>
                                <option value="mpesa">MPESA (KES only)</option>
                                <option value="card">Card (KES or USD)</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="email">Customer Email</label>
                            <input 
                                type="email" 
                                id="email" 
                                class="form-control" 
                                placeholder="customer@example.com" 
                                required 
                            />
                        </div>

                        <div class="form-group">
                            <label for="phone">Customer Phone (for MPESA)</label>
                            <input 
                                type="text" 
                                id="phone" 
                                class="form-control" 
                                placeholder="+254 700 123 456" 
                                required 
                            />
                        </div>

                        <div class="form-group">
                            <label for="account">Reference (e.g., Invoice ID)</label>
                            <input 
                                type="text" 
                                id="account" 
                                class="form-control" 
                                placeholder="INV-001" 
                                required 
                            />
                        </div>

                        <div class="form-group mt-3">
                            <button type="submit" class="btn btn-primary btn-block">
                                <i class="fas fa-paper-plane"></i> Send Payment Request
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Payment Status Modal -->
    <div id="paymentModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3><i class="fas fa-credit-card"></i> Payment Status</h3>
                <button class="modal-close">&times;</button>
            </div>
            <div class="modal-body" id="modalBody">
                <!-- Content will be populated by JavaScript -->
            </div>
            <div class="modal-footer">
                <button id="closeModal" class="btn btn-primary">Back to Transactions</button>
            </div>
        </div>
    </div>

    <!-- JavaScript to Handle API Call -->
    <script>
        let pollInterval;
        let pollTimeout;

        document.getElementById('paymentForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            const resultDiv = document.getElementById('result');
            resultDiv.innerHTML = '';

            // Show loading
            const submitBtn = e.target.querySelector('button');
            const originalText = submitBtn.innerHTML;
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';

            // Collect form data
            const amount = document.getElementById('amount').value;
            const currency = document.getElementById('currency').value;
            const method = document.getElementById('method').value;
            const email = document.getElementById('email').value;
            
            // Format phone number
            let phone = document.getElementById('phone').value;
            phone = phone.replace(/\D/g, '');
            if (phone.startsWith('0')) {
                phone = '254' + phone.substring(1);
            } else if (phone.startsWith('+254')) {
                phone = '254' + phone.substring(4);
            } else if (!phone.startsWith('254')) {
                phone = '254' + phone;
            }

            const account = document.getElementById('account').value;

            // Validate: MPESA only allows KES
            if (method === 'mpesa' && currency !== 'KES') {
                resultDiv.innerHTML = `
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle"></i> 
                        MPESA only supports KES. Please change the currency.
                    </div>`;
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
                return;
            }

            // Prepare payload
            const payload = {
                amount: parseFloat(amount),
                currency: currency,
                email: email,
                phone: phone,
                payment_method: method,
                account: account
            };

            try {
                // Initiate payment
                const response = await fetch('https://pay.macrapay.com/api/payment/initiate.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify(payload)
                });

                const data = await response.json();

                if (data.status === 'success') {
                    if (data.checkout_url) {
                        // Card payment - redirect to checkout
                        window.location.href = data.checkout_url;
                        return;
                    }

                    // MPESA payment - show modal and start polling
                    showPaymentModal('pending', 'Payment Request Sent', 'Please check your phone for the MPESA prompt.');
                    
                    // Start polling with timeout
                    startPolling(data.invoice_id);
                } else {
                    resultDiv.innerHTML = `
                        <div class="alert alert-danger">
                            <i class="fas fa-exclamation-circle"></i>
                            <strong>Error:</strong> ${data.error || 'Payment request failed.'}
                        </div>`;
                }
            } catch (err) {
                resultDiv.innerHTML = `
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle"></i>
                        <strong>Network Error:</strong> Unable to reach the payment server. Please try again.
                    </div>`;
                console.error('Fetch error:', err);
            } finally {
                // Restore button for MPESA (card payments redirect)
                if (!document.querySelector('#modalBody')) {
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalText;
                }
            }
        });

        function showPaymentModal(status, title, message) {
            const modal = document.getElementById('paymentModal');
            const modalBody = document.getElementById('modalBody');
            
            let icon = 'fa-info-circle';
            let iconClass = '';
            
            if (status === 'success') {
                icon = 'fa-check-circle';
                iconClass = 'success';
            } else if (status === 'failed') {
                icon = 'fa-times-circle';
                iconClass = 'failed';
            } else {
                icon = 'fa-clock';
            }
            
            modalBody.innerHTML = `
                <i class="fas ${icon} ${iconClass}"></i>
                <h3>${title}</h3>
                <p>${message}</p>
                ${status === 'pending' ? `
                    <div class="progress-container">
                        <div class="progress">
                            <div class="progress-bar" id="progressBar" style="width: 0%"></div>
                        </div>
                        <p id="progressText">Waiting for confirmation...</p>
                    </div>
                ` : ''}
            `;
            
            modal.style.display = 'flex';
        }

        function startPolling(invoiceId) {
            let attempts = 0;
            const maxAttempts = 12; // 30 seconds (5 seconds * 6)
            let progress = 0;
            
            // Update progress bar
            const progressBar = document.getElementById('progressBar');
            const progressText = document.getElementById('progressText');
            
            pollInterval = setInterval(async () => {
                try {
                    const statusResponse = await fetch(`https://pay.macrapay.com/api/payment/status.php?invoice_id=${invoiceId}`);
                    const statusData = await statusResponse.json();
                    
                    if (statusData.status === 'success') {
                        clearInterval(pollInterval);
                        clearTimeout(pollTimeout);
                        showPaymentModal('success', 'Payment Successful!', 'Your payment has been received successfully.');
                    } else if (statusData.status === 'failed') {
                        clearInterval(pollInterval);
                        clearTimeout(pollTimeout);
                        showPaymentModal('failed', 'Payment Failed', 'Your payment could not be processed. Please try again.');
                    } else {
                        attempts++;
                        progress = (attempts / maxAttempts) * 100;
                        progressBar.style.width = progress + '%';
                        progressText.textContent = `Waiting for confirmation... (${attempts}/${maxAttempts})`;
                        
                        if (attempts >= maxAttempts) {
                            clearInterval(pollInterval);
                            showPaymentModal('failed', 'Payment Timeout', 'Payment request timed out. Please check your transaction history.');
                        }
                    }
                } catch (err) {
                    attempts++;
                    progress = (attempts / maxAttempts) * 100;
                    progressBar.style.width = progress + '%';
                    progressText.textContent = `Waiting for confirmation... (${attempts}/${maxAttempts})`;
                    
                    if (attempts >= maxAttempts) {
                        clearInterval(pollInterval);
                        showPaymentModal('failed', 'Connection Error', 'Unable to check payment status. Please check your transaction history.');
                    }
                }
            }, 5000); // Poll every 5 seconds
            
            // Set timeout to stop polling after 30 seconds
            pollTimeout = setTimeout(() => {
                clearInterval(pollInterval);
                showPaymentModal('failed', 'Payment Timeout', 'Payment request timed out. Please check your transaction history.');
            }, 60000);
        }

        // Modal close handlers
        document.querySelector('.modal-close').addEventListener('click', () => {
            document.getElementById('paymentModal').style.display = 'none';
            clearInterval(pollInterval);
            clearTimeout(pollTimeout);
        });

        document.getElementById('closeModal').addEventListener('click', () => {
            document.getElementById('paymentModal').style.display = 'none';
            clearInterval(pollInterval);
            clearTimeout(pollTimeout);
            // Redirect to transactions page
            window.location.href = '../transactions/';
        });

        // Close modal on click outside
        window.addEventListener('click', (e) => {
            const modal = document.getElementById('paymentModal');
            if (e.target === modal) {
                modal.style.display = 'none';
                clearInterval(pollInterval);
                clearTimeout(pollTimeout);
            }
        });

        // Dynamic Currency Lock for MPESA
        const methodSelect = document.getElementById('method');
        const currencySelect = document.getElementById('currency');
        const currencyNote = document.getElementById('currencyNote');

        methodSelect.addEventListener('change', function () {
            if (this.value === 'mpesa') {
                currencySelect.value = 'KES';
                currencySelect.innerHTML = '<option value="KES">KES (Kenyan Shilling)</option>';
                currencyNote.style.display = 'block';
            } else {
                currencySelect.innerHTML = `
                    <option value="KES">KES (Kenyan Shilling)</option>
                    <option value="USD">USD (US Dollar)</option>
                `;
                currencyNote.style.display = 'none';
            }
        });
    </script>
</body>
</html>