<?php
// Initialize variables for error/success messages
$error = '';
$success = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Include database connection
    require_once '../conn/config.php';

    // Sanitize and get input
    $email = trim($_POST['email']);
    $phone = trim($_POST['phone']);
    $password = $_POST['password'];
    $user_type = $_POST['user_type'] ?? '';

    // Validate input
    if (empty($email) || empty($phone) || empty($password) || empty($user_type)) {
        $error = 'All fields are required.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = 'Please enter a valid email address.';
    } elseif (!preg_match('/^\+?[0-9]{10,15}$/', $phone)) {
        $error = 'Please enter a valid phone number (e.g., +254700123456).';
    } elseif (strlen($password) < 8) {
        $error = 'Password must be at least 8 characters long.';
    } elseif (!in_array($user_type, ['individual', 'company'])) {
        $error = 'Invalid account type selected.';
    } else {
        // Check if email already exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows > 0) {
            $error = 'An account with this email already exists.';
        } else {
            // Hash password
            $password_hash = password_hash($password, PASSWORD_DEFAULT);

            // Insert into users table
            $insert = $conn->prepare("
                INSERT INTO users (email, password_hash, phone, user_type, status, created_at) 
                VALUES (?, ?, ?, ?, 'pending', NOW())
            ");
            $insert->bind_param("ssss", $email, $password_hash, $phone, $user_type);

            if ($insert->execute()) {
                // Clear POST data to prevent resubmission
                $_POST = [];
                $success = 'Account created successfully! Please log in.';
            } else {
                $error = 'Something went wrong. Please try again.';
            }
            $insert->close();
        }
        $stmt->close();
    }
    $conn->close();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    
    <!-- Page Title -->
    <title>Create Your MacraPay Account | Secure Payment Gateway for Kenya</title>

    <!-- Meta Description -->
    <meta name="description" content="Sign up for MacraPay – Kenya’s trusted payment gateway. Accept MPESA and card payments securely for individuals and businesses. Fast onboarding, low fees, reliable service.">

    <!-- Keywords (optional) -->
    <meta name="keywords" content="MacraPay, payment gateway Kenya, accept MPESA, card payments Kenya, online payments, business payments, individual payments, signup, create account, secure payments">

    <!-- Author & Copyright -->
    <meta name="author" content="MacraPay Team">
    <meta name="copyright" content="MacraPay">
    <meta name="robots" content="index, follow"> <!-- Allow indexing when live -->

    <!-- Canonical URL (Prevent duplicate content) -->
    <link rel="canonical" href="https://macrapay.com/signup/" />

    <!-- Open Graph / Social Sharing (Facebook, WhatsApp, LinkedIn) -->
    <meta property="og:title" content="Sign Up | MacraPay - Accept MPESA & Card Payments">
    <meta property="og:description" content="Create your free MacraPay account and start accepting secure payments via MPESA and cards in Kenya. Built for individuals and businesses.">
    <meta property="og:type" content="website">
    <meta property="og:url" content="https://macrapay.com/signup/">
    <meta property="og:image" content="https://macrapay.com/images/og-signup.jpg">
    <meta property="og:image:alt" content="MacraPay Signup - Secure Payment Gateway for Kenya">
    <meta property="og:site_name" content="MacraPay">
    <meta property="og:locale" content="en_KE">

    <!-- Twitter Card -->
    <meta name="twitter:card" content="summary_large_image">
    <meta name="twitter:title" content="Sign Up | MacraPay - Accept MPESA & Card Payments">
    <meta name="twitter:description" content="Join MacraPay today and receive payments via MPESA and cards with ease. Trusted by businesses and individuals across Kenya.">
    <meta name="twitter:image" content="https://macrapay.com/images/og-signup.jpg">
    <meta name="twitter:image:alt" content="Create your MacraPay account in minutes">
    <meta name="twitter:site" content="@macrapay">
    <meta name="twitter:creator" content="@macrapay">

    <!-- Favicon -->
    <link rel="icon" type="image/png" href="../images/logo.png" sizes="32x32" />
    <link rel="apple-touch-icon" href="../images/logo.png" />

    <!-- Preload Critical Assets (Optional) -->
    <link rel="preload" as="style" href="../css/styles.css" />
    <link rel="preload" as="image" href="../images/logo.png" />

    <!-- Font Awesome CDN -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" integrity="sha512-..." crossorigin="anonymous" referrerpolicy="no-referrer" />

    <!-- Main Stylesheet -->
    <link rel="stylesheet" href="../css/styles.css" />

    <!-- Optional: Theme Color for Chrome, Firefox, Safari -->
    <meta name="theme-color" content="#007bff">
</head>
<body>

    <div class="nav">
        <div class="container">
            <a href="/" class="nav-logo">
                <img src="../images/logo.png" alt="MacraPay Logo"> MacraPay
            </a>
            <a href="../login/" class="btn btn-secondary"><i class="fas fa-sign-in-alt"></i> Login</a>
        </div>
    </div>

    <div class="container">
        <div class="card">
            <img src="../images/logo.png" alt="MacraPay Logo" class="logo" />
            <h2 class="card-title"><i class="fas fa-user-plus"></i> Create Your Account</h2>

            <!-- Display Messages -->
            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle"></i> <?= htmlspecialchars($error) ?>
                </div>
            <?php endif; ?>

            <?php if ($success): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i> <?= htmlspecialchars($success) ?>
                </div>
            <?php endif; ?>

            <form id="signupForm" action="" method="POST">
                <div class="form-group input-icon">
                    <label for="email"><i class="fas fa-envelope"></i> Email Address</label>
                    <input 
                        type="email" 
                        id="email" 
                        name="email" 
                        placeholder="you@example.com" 
                        value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" 
                        required 
                    />
                </div>

                <div class="form-group input-icon">
                    <label for="phone"><i class="fas fa-phone"></i> Phone Number</label>
                    <input 
                        type="tel" 
                        id="phone" 
                        name="phone" 
                        placeholder="+254 700 123 456" 
                        value="<?= htmlspecialchars($_POST['phone'] ?? '') ?>" 
                        required 
                    />
                </div>

                <div class="form-group input-icon">
                    <label for="password"><i class="fas fa-lock"></i> Password</label>
                    <input 
                        type="password" 
                        id="password" 
                        name="password" 
                        placeholder="••••••••" 
                        minlength="8" 
                        required 
                    />
                </div>

                <div class="form-group">
                    <label for="user_type"><i class="fas fa-building"></i> Account Type</label>
                    <select id="user_type" name="user_type" required>
                        <option value="">Choose account type</option>
                        <option value="individual" <?= (isset($_POST['user_type']) && $_POST['user_type'] === 'individual') ? 'selected' : '' ?>>
                            Individual
                        </option>
                        <option value="company" <?= (isset($_POST['user_type']) && $_POST['user_type'] === 'company') ? 'selected' : '' ?>>
                            Company / Business
                        </option>
                    </select>
                </div>

                <div class="form-group mt-3">
                    <button type="submit" class="btn btn-primary btn-block">
                        <span class="btn-text"><i class="fas fa-user-plus"></i> Sign Up</span>
                        <span class="loader" id="loader"></span>
                    </button>
                </div>
            </form>

            <p class="text-center mt-2">
                <small>
                    By signing up, you agree to our <a href="/terms">Terms</a> and <a href="/privacy">Privacy Policy</a>.
                </small>
            </p>
        </div>

        <p class="text-center">
            Already have an account? <a href="../login/"><i class="fas fa-sign-in-alt"></i> Log in</a>
        </p>
    </div>

    <!-- Form UX: Show loader on submit -->
    <script>
        document.getElementById("signupForm").addEventListener("submit", function () {
            const btnText = document.querySelector(".btn-text");
            const loader = document.getElementById("loader");
            btnText.style.display = "none";
            loader.style.display = "inline-block";
        });
    </script>

</body>
</html>