<?php
// Start session and protect
session_start();
if (!isset($_SESSION['user_id']) || !isset($_SESSION['logged_in']) || !$_SESSION['logged_in']) {
    header("Location: ../login/");
    exit();
}

// Include DB connection
require_once '../conn/config.php';

// Initialize variables
$transactions = [];
$error = '';

// Fixed query: Removed `status` (it doesn't exist in the table)
$query = "
    SELECT 
        id,
        amount,
        currency,
        type,
        description,
        created_at
    FROM transactions 
    WHERE user_id = ? 
    ORDER BY created_at DESC
";

$stmt = $conn->prepare($query);

if ($stmt === false) {
    $error = "Database prepare error: " . $conn->error;
} else {
    $user_id = $_SESSION['user_id'];
    $stmt->bind_param("i", $user_id);

    if ($stmt->execute()) {
        $result = $stmt->get_result();
        if ($result && $result->num_rows > 0) {
            while ($row = $result->fetch_assoc()) {
                // Add a default status based on type if needed
                $row['status'] = 'completed'; // Since there's no status column
                $transactions[] = $row;
            }
        } else {
            $error = 'No transactions found.';
        }
    } else {
        $error = "Execute failed: " . $stmt->error;
    }

    $stmt->close();
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
    <title>Transactions | MacraPay</title>

    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" />

    <!-- Custom Styles -->
    <link rel="stylesheet" href="../css/styles.css" />
    <!-- Favicon -->
    <link rel="icon" type="image/png" href="../images/logo.png" />
</head>
<body>
    <!-- Include Sidebar -->
    <?php include '../includes/sidebar.php'; ?>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Include Header -->
        <?php include '../includes/header.php'; ?>

        <div class="content">
            <div class="container">
                <h1><i class="fas fa-exchange-alt"></i> Transactions</h1>
                <p class="mb-2">Complete financial history: payments, payouts, refunds, and fees.</p>

                <!-- Search & Filter -->
                <div class="form-row mb-2">
                    <div class="form-col">
                        <label for="searchTransactions"><i class="fas fa-search"></i> Search</label>
                        <input 
                            type="text" 
                            id="searchTransactions" 
                            class="form-control" 
                            placeholder="Search by description, amount..." 
                        />
                    </div>
                    <div class="form-col">
                        <label for="filterType">Filter by Type</label>
                        <select id="filterType" class="form-control">
                            <option value="">All Types</option>
                            <option value="payment">Payment</option>
                            <option value="payout">Payout</option>
                            <option value="refund">Refund</option>
                            <option value="fee">Fee</option>
                        </select>
                    </div>
                </div>

                <?php if ($error): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle"></i> <?= htmlspecialchars($error) ?>
                    </div>
                <?php elseif (empty($transactions)): ?>
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i> No transactions found.
                    </div>
                <?php else: ?>
                    <div class="table-responsive">
                        <table id="transactionsTable">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Type</th>
                                    <th>Description</th>
                                    <th>Amount</th>
                                    <!-- Removed Status column -->
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($transactions as $t): ?>
                                <tr data-type="<?= htmlspecialchars($t['type']) ?>"
                                    data-search="<?= strtolower($t['description'] . ' ' . $t['amount']) ?>">
                                    <td>
                                        <?= date('M j, Y', strtotime($t['created_at'])) ?><br>
                                        <small><?= date('h:i A', strtotime($t['created_at'])) ?></small>
                                    </td>
                                    <td>
                                        <span class="badge-type type-<?= htmlspecialchars($t['type']) ?>">
                                            <?= ucfirst(htmlspecialchars($t['type'])) ?>
                                        </span>
                                    </td>
                                    <td><?= htmlspecialchars($t['description'] ?? 'N/A') ?></td>
                                    <td>
                                        <strong class="<?php 
                                            echo in_array($t['type'], ['payout', 'refund', 'fee']) ? 'text-danger' : 'text-success'; 
                                        ?>">
                                            <?php echo in_array($t['type'], ['payout', 'refund', 'fee']) ? '-' : '+'; ?>
                                            <?= htmlspecialchars($t['currency']) ?> <?= number_format($t['amount'], 2) ?>
                                        </strong>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script>
        // 🔍 Live Search
        document.getElementById('searchTransactions')?.addEventListener('keyup', function () {
            const term = this.value.toLowerCase();
            document.querySelectorAll('#transactionsTable tbody tr').forEach(row => {
                const search = row.dataset.search || '';
                row.style.display = search.includes(term) ? '' : 'none';
            });
        });

        // 🧩 Filter by Type
        document.getElementById('filterType')?.addEventListener('change', function () {
            const selected = this.value;
            document.querySelectorAll('#transactionsTable tbody tr').forEach(row => {
                const type = row.dataset.type;
                const matches = !selected || type === selected;
                row.style.display = matches ? '' : 'none';
            });
        });
    </script>

    <!-- Styles -->
    <style>
        .badge-type {
            padding: 0.3rem 0.6rem;
            border-radius: 50px;
            font-size: 0.8rem;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .type-payment { background: #d4edda; color: #155724; }
        .type-payout  { background: #fff3cd; color: #856404; }
        .type-refund  { background: #f8d7da; color: #721c24; }
        .type-fee     { background: #d1ecf1; color: #0c5460; }
        .text-success { color: #28a745; }
        .text-danger  { color: #dc3545; }
    </style>
</body>
</html>