<?php
// api/payment/callback.php

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit();
}

$payload = file_get_contents('php://input');
$data    = json_decode($payload, true);

file_put_contents('callback.log', date('Y-m-d H:i:s') . " - " . $payload . PHP_EOL, FILE_APPEND);

if (!isset($data['invoice_id'], $data['state'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid payload']);
    exit();
}

require_once '../../conn/config.php';

$invoice_id = $data['invoice_id'];
$status     = $data['state'];

$new_status = match (strtolower($status)) {
    'complete', 'completed', 'paid' => 'success',
    'failed', 'cancelled'           => 'failed',
    default                         => 'pending'
};

$conn->begin_transaction();

try {
    // 1️⃣  Update the transactions row
    $update_stmt = $conn->prepare(
        "UPDATE transactions 
         SET status = ? 
         WHERE description LIKE CONCAT('%Invoice: ', ?, '%')"
    );
    $update_stmt->bind_param('ss', $new_status, $invoice_id);
    $update_stmt->execute();
    $update_stmt->close();

    if ($new_status === 'success') {
        // 2️⃣  Fetch the transaction row
        $trans_stmt = $conn->prepare(
            "SELECT id, user_id, amount, currency
             FROM transactions
             WHERE description LIKE CONCAT('%Invoice: ', ?, '%')
             LIMIT 1"
        );
        $trans_stmt->bind_param('s', $invoice_id);
        $trans_stmt->execute();
        $trans_res = $trans_stmt->get_result();
        if ($trans_res->num_rows === 0) {
            throw new Exception('Transaction not found for invoice: ' . $invoice_id);
        }
        $txn = $trans_res->fetch_assoc();
        $trans_stmt->close();

        $user_id   = $txn['user_id'];
        $gross     = $txn['amount'];
        $currency  = $txn['currency'];
        $charge    = round($gross * 0.02, 2);   // 2 %
        $net       = $gross - $charge;

        // 3️⃣  Insert into payments (full gross amount)
        $pay_stmt = $conn->prepare(
            "INSERT INTO payments (user_id, amount, currency, payment_method, reference, status)
             VALUES (?, ?, ?, 'mpesa', ?, ?)"
        );
        $pay_stmt->bind_param('idsss', $user_id, $gross, $currency, $invoice_id, $new_status);
        $pay_stmt->execute();
        $pay_stmt->close();

        // 4️⃣  Insert the charge record
        $desc_charge = "Transaction charge on invoice {$invoice_id}";
        $charge_stmt = $conn->prepare(
            "INSERT INTO charges (user_id, transaction_id, amount, currency, description, status)
             VALUES (?, ?, ?, ?, ?, 'processed')"
        );
        $charge_stmt->bind_param('iidss', $user_id, $txn['id'], $charge, $currency, $desc_charge);
        $charge_stmt->execute();
        $charge_stmt->close();

        // 5️⃣  Compute & update user balance
        $bal_stmt = $conn->prepare(
            "SELECT balance_after
             FROM transactions
             WHERE user_id = ? AND balance_after IS NOT NULL
             ORDER BY created_at DESC
             LIMIT 1"
        );
        $bal_stmt->bind_param('i', $user_id);
        $bal_stmt->execute();
        $bal_res = $bal_stmt->get_result();
        $current_balance = $bal_res->num_rows ? (float) $bal_res->fetch_assoc()['balance_after'] : 0.0;
        $bal_stmt->close();

        $new_balance = $current_balance + $net;

        $upd_bal_stmt = $conn->prepare(
            "UPDATE transactions SET balance_after = ? WHERE id = ?"
        );
        $upd_bal_stmt->bind_param('di', $new_balance, $txn['id']);
        $upd_bal_stmt->execute();
        $upd_bal_stmt->close();
    }

    $conn->commit();
} catch (Exception $e) {
    $conn->rollback();
    error_log("Callback processing error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Internal server error']);
    exit();
}

$conn->close();
echo json_encode(['received' => true]);