<?php
// api/payment/initiate.php

// Allow CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

session_start();

// Load config
require_once 'config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

$input = json_decode(file_get_contents('php://input'), true);

$amount         = $input['amount']        ?? null;
$currency       = strtoupper($input['currency'] ?? 'KES');
$email          = $input['email']         ?? null;
$phone          = $input['phone']         ?? null;
$account        = $input['account']       ?? null;
$payment_method = strtolower($input['payment_method'] ?? '');

// Validation
if (!$amount || !$email || !$phone || !$account || !$payment_method) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing required fields']);
    exit();
}

if (!is_numeric($amount) || $amount <= 0) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid amount']);
    exit();
}

if ($payment_method === 'mpesa' && $currency !== 'KES') {
    http_response_code(400);
    echo json_encode(['error' => 'MPESA only supports KES.']);
    exit();
}

if ($payment_method === 'card' && !in_array($currency, ['KES', 'USD'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Card payments only support KES or USD.']);
    exit();
}

// Generate unique reference
$reference = 'INV_' . uniqid();

try {
    // Prepare payload
    $payload = [
        'public_key'   => INTASEND_PUBLISHABLE_KEY,
        'amount'       => $amount,
        'currency'     => $currency,
        'method'       => strtoupper($payment_method),
        'phone_number' => $phone,
        'email'        => $email,
        'account'      => $account,
        'api_ref'      => $reference,
        'redirect_url' => INTASEND_CALLBACK_URL,
        'live'         => !INTASEND_TEST_MODE
    ];

    // IntaSend API endpoint
    $url = INTASEND_TEST_MODE
        ? 'https://sandbox.intasend.com/api/v1/payment/mpesa-stk-push/'
        : 'https://payment.intasend.com/api/v1/payment/mpesa-stk-push/';

    if ($payment_method === 'card') {
        $url = INTASEND_TEST_MODE
            ? 'https://sandbox.intasend.com/api/v1/payment/card/'
            : 'https://payment.intasend.com/api/v1/payment/card/';
    }

    // Initialize cURL
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $url,
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => json_encode($payload),
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'Authorization: Bearer ' . INTASEND_API_KEY
        ],
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 30
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError) {
        throw new Exception("cURL Error: " . $curlError);
    }

    $data = json_decode($response, true);

    // Handle MPESA response
    if (isset($data['invoice']['invoice_id'], $data['invoice']['state']) &&
        $data['invoice']['state'] === 'PENDING') {

        require_once '../../conn/config.php';

        $user_id    = $_SESSION['user_id'] ?? 1;
        $invoice_id = $data['invoice']['invoice_id'];
        $type       = 'payment';
        $status     = 'pending';
        $description = "Payment initiated via IntaSend - Invoice: " . $invoice_id;

        $stmt = $conn->prepare(
            "INSERT INTO transactions 
             (user_id, amount, currency, type, description, status)
             VALUES (?, ?, ?, ?, ?, ?)"
        );
        $stmt->bind_param("idssss", $user_id, $amount, $currency, $type, $description, $status);
        $stmt->execute();
        $stmt->close();
        $conn->close();

        echo json_encode([
            'status'     => 'success',
            'invoice_id' => $invoice_id,
            'message'    => 'Payment initiated. Waiting for confirmation...'
        ]);
    }
    // Handle Card response
    elseif (isset($data['url'])) {

        require_once '../../conn/config.php';

        $user_id    = $_SESSION['user_id'] ?? 1;
        $invoice_id = $data['invoice']['invoice_id'] ?? $reference;
        $type       = 'payment';
        $status     = 'pending';
        $description = "Card payment initiated via IntaSend - Invoice: " . $invoice_id;

        $stmt = $conn->prepare(
            "INSERT INTO transactions 
             (user_id, amount, currency, type, description, status)
             VALUES (?, ?, ?, ?, ?, ?)"
        );
        $stmt->bind_param("idssss", $user_id, $amount, $currency, $type, $description, $status);
        $stmt->execute();
        $stmt->close();
        $conn->close();

        echo json_encode([
            'status'       => 'success',
            'invoice_id'   => $invoice_id,
            'checkout_url' => $data['url'],
            'message'      => 'Redirecting to card payment page...'
        ]);
    } else {
        error_log("IntaSend API Error: " . $response);
        http_response_code($httpCode ?: 500);
        echo json_encode([
            'error'   => 'Payment initiation failed',
            'details' => $data ?? $response
        ]);
    }
} catch (Exception $e) {
    error_log("Payment Initiation Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Payment setup failed. Please try again.']);
}