<?php
// api/payouts/process.php

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit();
}

session_start();
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit();
}

$input = json_decode(file_get_contents('php://input'), true);

$amount = $input['amount'] ?? null;
$fee = $input['fee'] ?? null;
$net_amount = $input['net_amount'] ?? null;
$phone = $input['phone'] ?? null;

// Validation
if (!$amount || !$fee || !$net_amount || !$phone) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing required fields']);
    exit();
}

if (!is_numeric($amount) || $amount <= 0) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid amount']);
    exit();
}

if (!preg_match('/^254\d{9}$/', $phone)) {
    http_response_code(400);
    echo json_encode(['error' => 'Invalid phone number format. Use 254XXXXXXXXX']);
    exit();
}

$user_id = $_SESSION['user_id'];

try {
    require_once '../../conn/config.php';
    
    // Start transaction
    $conn->begin_transaction();
    
    // Check user balance
    $balance_stmt = $conn->prepare("SELECT balance_after FROM transactions WHERE user_id = ? AND balance_after IS NOT NULL ORDER BY created_at DESC LIMIT 1");
    $balance_stmt->bind_param("i", $user_id);
    $balance_stmt->execute();
    $balance_result = $balance_stmt->get_result();
    
    $current_balance = 0;
    if ($balance_result->num_rows > 0) {
        $row = $balance_result->fetch_assoc();
        $current_balance = $row['balance_after'];
    }
    $balance_stmt->close();
    
    if ($amount > $current_balance) {
        throw new Exception('Insufficient balance');
    }
    
    // Insert into payouts table
    $payout_stmt = $conn->prepare("INSERT INTO payouts (user_id, amount, fee, net_amount, destination_type, destination_account, status) VALUES (?, ?, ?, ?, 'mpesa', ?, 'pending')");
    $payout_stmt->bind_param("iddds", $user_id, $amount, $fee, $net_amount, $phone);
    $payout_stmt->execute();
    $payout_id = $conn->insert_id;
    $payout_stmt->close();
    
    // Insert into transactions table (negative amount for withdrawal)
    $description = "Withdrawal to MPESA " . $phone;
    // Calculate new balance (subtract total amount including fee)
    $new_balance = $current_balance - $amount;
    
    $trans_stmt = $conn->prepare("INSERT INTO transactions (user_id, amount, currency, type, description, status, balance_after) VALUES (?, ?, 'KES', 'payout', ?, 'pending', ?)");
    $trans_stmt->bind_param("idssd", $user_id, $amount, $description, $new_balance);
    $trans_stmt->execute();
    $trans_stmt->close();
    
    // Here you would integrate with IntaSend MPESA payout API
    // For now, we'll simulate success
    
    // Update payout status to sent
    $update_stmt = $conn->prepare("UPDATE payouts SET status = 'sent' WHERE id = ?");
    $update_stmt->bind_param("i", $payout_id);
    $update_stmt->execute();
    $update_stmt->close();
    
    // Update transaction status to success
    $update_trans_stmt = $conn->prepare("UPDATE transactions SET status = 'success' WHERE user_id = ? AND type = 'payout' AND status = 'pending' ORDER BY created_at DESC LIMIT 1");
    $update_trans_stmt->bind_param("i", $user_id);
    $update_trans_stmt->execute();
    $update_trans_stmt->close();
    
    $conn->commit();
    
    echo json_encode([
        'status' => 'success',
        'payout_id' => 'PAYOUT_' . $payout_id,
        'message' => 'Withdrawal initiated successfully'
    ]);
    
} catch (Exception $e) {
    $conn->rollback();
    error_log("Payout error: " . $e->getMessage());
    
    if ($e->getMessage() === 'Insufficient balance') {
        http_response_code(400);
        echo json_encode(['error' => 'Insufficient balance for this withdrawal']);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to process withdrawal. Please try again.']);
    }
    exit();
}

$conn->close();
?>